<?php
defined( 'ABSPATH' ) || exit;

class HRM_Dashboard {

    public static function init() {
        add_action( 'admin_post_hrm_save_settings', [ __CLASS__, 'handle_save_settings' ] );
        add_action( 'wp_ajax_hrm_save_settings', [ __CLASS__, 'handle_save_settings' ] );
    }

    public static function handle_save_settings() {
        check_admin_referer( 'hrm_save_settings_nonce' );
    
        if ( ! current_user_can( 'manage_options' ) ) {
            HRM_Security::log( 'access_denied', 'Unauthorized settings save attempt.' );
            wp_die( 'Unauthorized' );
        }
        
        // Re-hash file on settings save
        if ( defined('HRM_PLUGIN_FILE') ) {
             update_option( 'hrm_core_file_hash', hash_file( 'sha256', HRM_PLUGIN_FILE ) );
        }
        HRM_Security::log( 'system', 'Integrity hash updated by admin.' );
    
        if ( isset( $_POST['hrm_notification_email'] ) ) {
            $emails = explode( ',', $_POST['hrm_notification_email'] );
            $valid_emails = [];
            foreach ( $emails as $email ) {
                $email = sanitize_email( trim( $email ) );
                if ( is_email( $email ) ) {
                    $valid_emails[] = $email;
                }
            }
            update_option( 'hrm_notification_email', implode( ',', $valid_emails ) );
        }

        // Save Company Logo
        if ( isset( $_POST['hrm_company_logo_id'] ) ) {
            $logo_id = absint( $_POST['hrm_company_logo_id'] );
            update_option( 'hrm_company_logo_id', $logo_id );
        }

        // IP Configuration is now managed via the dedicated IP Configuration tab

        if ( wp_doing_ajax() ) {
            wp_send_json_success( 'Settings saved successfully.' );
        } else {
            wp_redirect( admin_url( 'admin.php?page=hrm-settings&message=saved' ) );
            exit;
        }
    }

    public static function render_settings_page() {
        $active_tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : 'general';
        $emails_string = get_option( 'hrm_notification_email', get_option( 'admin_email' ) );
        $logo_id = get_option( 'hrm_company_logo_id', 0 );
        $logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'medium' ) : '';
        
        // Enqueue media uploader
        wp_enqueue_media();
        ?>
        <?php
        $wrap_class = in_array($active_tab, ['security', 'ip-config']) ? 'wrap hrm-wrap hrm-modern-dashboard hrm-full-width' : 'wrap hrm-wrap hrm-modern-dashboard';
        ?>
        <div class="<?php echo esc_attr($wrap_class); ?>">
            <div class="hrm-header-section" style="margin-bottom: 30px;">
                <div class="hrm-header-left">
                    <h1>Settings & Security</h1>
                    <p>Configure plugin parameters, manageable IPs, and security logs.</p>
                </div>
                <div class="hrm-header-right">
                    <div class="hrm-tabs-wrapper">
                        <a href="?page=hrm-settings&tab=general" class="hrm-tab-item <?php echo $active_tab === 'general' ? 'active' : ''; ?>">
                            <span class="dashicons dashicons-admin-settings"></span> General
                        </a>
                        <a href="?page=hrm-settings&tab=ip-config" class="hrm-tab-item <?php echo $active_tab === 'ip-config' ? 'active' : ''; ?>">
                            <span class="dashicons dashicons-admin-network"></span> IP Configuration
                        </a>
                        <a href="?page=hrm-settings&tab=security" class="hrm-tab-item <?php echo $active_tab === 'security' ? 'active' : ''; ?>">
                            <span class="dashicons dashicons-shield"></span> Security Log
                        </a>
                    </div>
                </div>
            </div>

            <?php if ( isset( $_GET['message'] ) && $_GET['message'] === 'saved' ) : ?>
                <div class="hrm-toast-notification">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <span>Settings saved successfully!</span>
                    <button class="hrm-toast-close" onclick="this.parentElement.remove()">&times;</button>
                </div>
            <?php endif; ?>

            <div class="hrm-settings-container">
            <?php if ( $active_tab === 'general' ) : ?>
                <div class="hrm-card hrm-settings-card">
                    <div class="card-header-simple">
                        <h2>General Configuration</h2>
                        <p>Manage notification preferences and access controls.</p>
                    </div>
                    
                    <form id="hrm-dashboard-settings-form" method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                        <input type="hidden" name="action" value="hrm_save_settings">
                        <?php wp_nonce_field( 'hrm_save_settings_nonce' ); ?>
                        
                        <div class="hrm-section-block">
                            <div class="hrm-form-group full-width">
                                <label>Notification Emails</label>
                                <p class="description">Enter comma-separated emails to receive system notifications (e.g. leave requests, daily reports).</p>
                                <div class="hrm-input-with-icon">
                                    <span class="dashicons dashicons-email-alt"></span>
                                    <input type="text" name="hrm_notification_email" value="<?php echo esc_attr( $emails_string ); ?>" placeholder="admin@example.com, hr@example.com" style="padding-left: 36px;">
                                </div>
                            </div>
                        </div>

                        <div class="hrm-section-block">
                            <div class="hrm-form-group full-width">
                                <label>Company Logo</label>
                                <p class="description">Upload your company logo to display on the Employee Login screen and Employee Portal header. Recommended size: 200x60 pixels.</p>
                                
                                <div class="hrm-logo-upload-wrapper" style="margin-top: 12px;">
                                    <input type="hidden" name="hrm_company_logo_id" id="hrm_company_logo_id" value="<?php echo esc_attr( $logo_id ); ?>">
                                    
                                    <div id="hrm-logo-preview" class="hrm-logo-preview" style="margin-bottom: 16px; padding: 20px; background: #F9FAFB; border-radius: 12px; border: 2px dashed #E5E7EB; text-align: center; min-height: 100px; display: flex; align-items: center; justify-content: center;">
                                        <?php if ( $logo_url ) : ?>
                                            <img src="<?php echo esc_url( $logo_url ); ?>" alt="Company Logo" style="max-width: 200px; max-height: 80px; width: auto; height: auto;">
                                        <?php else : ?>
                                            <span style="color: #9CA3AF; font-size: 14px;"><span class="dashicons dashicons-format-image" style="font-size: 32px; width: 32px; height: 32px; display: block; margin: 0 auto 8px;"></span>No logo uploaded</span>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div style="display: flex; gap: 12px;">
                                        <button type="button" id="hrm-upload-logo-btn" class="hrm-btn-outline" style="padding: 10px 20px;">
                                            <span class="dashicons dashicons-upload" style="vertical-align: middle; margin-right: 6px;"></span>
                                            <?php echo $logo_url ? 'Change Logo' : 'Upload Logo'; ?>
                                        </button>
                                        <?php if ( $logo_url ) : ?>
                                            <button type="button" id="hrm-remove-logo-btn" class="hrm-btn-outline" style="padding: 10px 20px; color: #EF4444; border-color: #FCA5A5;">
                                                <span class="dashicons dashicons-trash" style="vertical-align: middle; margin-right: 6px;"></span>
                                                Remove Logo
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="hrm-section-block">
                            <div class="hrm-form-group full-width">
                                <label>Attendance IP Restrictions</label>
                                <p class="description">
                                    Control which IP addresses are allowed for employee check-ins.
                                </p>
                                <div class="hrm-ip-config-link">
                                    <span class="dashicons dashicons-admin-network" style="color: var(--primary-color);"></span>
                                    <span>IP-based access control is now managed in the dedicated IP Configuration tab.</span>
                                    <a href="?page=hrm-settings&tab=ip-config" class="hrm-btn-outline btn-sm" style="margin-left: auto;">
                                        <span class="dashicons dashicons-arrow-right-alt"></span> Go to IP Configuration
                                    </a>
                                </div>
                            </div>
                        </div>

                        <div class="hrm-form-actions-footer">
                            <button type="submit" class="hrm-btn-primary hrm-btn-large">
                                <span class="dashicons dashicons-saved"></span> Save Changes
                            </button>
                        </div>
                    </form>
                </div>
            <?php elseif ( $active_tab === 'ip-config' ) : ?>
                <!-- IP Configuration Tab -->
                <?php HRM_IP_Config::render_ip_config_tab(); ?>
            <?php else : ?>
                <!-- Security Tab -->
                <div class="hrm-card hrm-settings-card full-width-card">
                    <div class="card-header-simple">
                        <h2>Security Audit Log</h2>
                        <p>Detailed tracking of all critical system events and access attempts.</p>
                    </div>
                    
                    <?php
                    global $wpdb;

                    // Week Pagination Logic
                    $week_offset = isset($_GET['week']) ? absint($_GET['week']) : 0;
                    
                    // Employee Filter
                    $employee_filter = isset($_GET['employee_filter']) ? intval($_GET['employee_filter']) : 0;
                    
                    // Calculate Week Start (Monday) and End (Sunday)
                    // If week_offset = 0 (This week), Start = This Week's Monday.
                    // If week_offset = 1 (Last week), Start = Last Week's Monday.
                    
                    $current_timestamp = current_time('timestamp');
                    $this_week_monday = strtotime('monday this week', $current_timestamp);
                    
                    // Target Week Start
                    $target_week_start = strtotime("- $week_offset weeks", $this_week_monday);
                    // Target Week End (Sunday 23:59:59)
                    $target_week_end = strtotime("+6 days 23:59:59", $target_week_start);
                    
                    $start_date_sql = date('Y-m-d H:i:s', $target_week_start);
                    $end_date_sql = date('Y-m-d H:i:s', $target_week_end);
                    
                    // Build query based on employee filter
                    if ( $employee_filter ) {
                        $logs = $wpdb->get_results( 
                            $wpdb->prepare(
                                "SELECT * FROM {$wpdb->prefix}hrm_security_logs 
                                 WHERE created_at BETWEEN %s AND %s 
                                 AND user_id = %d
                                 ORDER BY created_at DESC",
                                $start_date_sql, $end_date_sql, $employee_filter
                            )
                        );
                    } else {
                        $logs = $wpdb->get_results( 
                            $wpdb->prepare(
                                "SELECT * FROM {$wpdb->prefix}hrm_security_logs 
                                 WHERE created_at BETWEEN %s AND %s 
                                 ORDER BY created_at DESC",
                                $start_date_sql, $end_date_sql
                            )
                        );
                    }
                    ?>
                    
                    <!-- Employee Filter Dropdown -->
                    <div class="hrm-filter-bar hrm-card" style="margin-bottom: 20px; padding: 15px 20px; display: flex; align-items: center; gap: 15px; flex-wrap: wrap;">
                        <div style="display: flex; align-items: center; gap: 8px; flex: 1; min-width: 200px;">
                            <span class="dashicons dashicons-admin-users" style="color: #667eea; font-size: 20px;"></span>
                            <label for="security-employee-filter" style="font-weight: 600; color: #374151; margin: 0; white-space: nowrap;">
                                Filter by Employee:
                            </label>
                        </div>
                        <select id="security-employee-filter" class="employee-select-dropdown" style="padding: 8px 12px; border-radius: 6px; border: 1px solid #d1d5db; min-width: 200px; flex: 1;">
                            <option value="">All Employees</option>
                            <?php
                            // Get all employees for the dropdown - using custom post type (more reliable)
                            $employees_posts = get_posts([
                                'post_type' => 'hrm_employee', 
                                'posts_per_page' => -1,
                                'post_status' => ['publish', 'draft'],
                                'orderby' => 'title',
                                'order' => 'ASC'
                            ]);
                            
                            // If we have employees from post type, use those
                            if ( !empty($employees_posts) ) {
                                $added_users = []; // Track which user IDs we've already added
                                
                                foreach ( $employees_posts as $emp_post ) {
                                    $user_id = $emp_post->post_author;
                                    
                                    // Skip if we've already added this user (prevents duplicates)
                                    if ( in_array($user_id, $added_users) ) {
                                        continue;
                                    }
                                    
                                    $selected = ( $employee_filter == $user_id ) ? 'selected' : '';
                                    echo '<option value="' . esc_attr( $user_id ) . '" ' . $selected . '>' . esc_html( $emp_post->post_title ) . '</option>';
                                    
                                    // Mark this user as added
                                    $added_users[] = $user_id;
                                }
                            } else {
                                // Fallback: Query users with hrm_employee role
                                $users = get_users([
                                    'role' => 'hrm_employee',
                                    'orderby' => 'display_name',
                                    'order' => 'ASC'
                                ]);
                                
                                foreach ( $users as $user ) {
                                    $selected = ( $employee_filter == $user->ID ) ? 'selected' : '';
                                    echo '<option value="' . esc_attr( $user->ID ) . '" ' . $selected . '>' . esc_html( $user->display_name ) . '</option>';
                                }
                            }
                            ?>
                        </select>
                        <button type="button" id="apply-security-filter" class="hrm-btn-primary" style="padding: 10px 20px; white-space: nowrap;">
                            <span class="dashicons dashicons-filter" style="vertical-align: middle;"></span>
                            Apply Filter
                        </button>
                        <?php if ( $employee_filter ): ?>
                            <a href="?page=hrm-settings&tab=security&week=<?php echo $week_offset; ?>" class="hrm-btn-outline" style="padding: 10px 20px; text-decoration: none; white-space: nowrap;">
                                <span class="dashicons dashicons-no-alt" style="vertical-align: middle;"></span>
                                Clear Filter
                            </a>
                        <?php endif; ?>
                    </div>
                    
                    <div class="hrm-table-responsive">
                        <table class="hrm-table">
                            <thead>
                                <tr>
                                    <th>Date & Time</th>
                                    <th>Event Type</th>
                                    <th>User</th>
                                    <th>IP Address</th>
                                    <th>Details</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ( $logs ) : ?>
                                    <?php foreach ( $logs as $log ) : 
                                        $user_info = $log->user_id ? get_user_by( 'id', $log->user_id )->user_login : 'Guest';
                                        $badge_class = 'badge-blue';
                                        if($log->event_type === 'access_denied') $badge_class = 'badge-red';
                                        elseif($log->event_type === 'system') $badge_class = 'badge-yellow';
                                    ?>
                                    <tr>
                                        <td>
                                            <span style="font-weight:600; color:#374151;"><?php echo date( 'M d, Y', strtotime( $log->created_at ) ); ?></span>
                                            <span style="display:block; font-size:12px; color:#6B7280;"><?php echo date( 'H:i:s', strtotime( $log->created_at ) ); ?></span>
                                        </td>
                                        <td><span class="hrm-badge <?php echo $badge_class; ?>"><?php echo esc_html( ucfirst(str_replace('_', ' ', $log->event_type)) ); ?></span></td>
                                        <td>
                                            <div class="hrm-user-cell">
                                                <div class="hrm-avatar-sm" style="width:24px; height:24px; font-size:10px;">
                                                    <?php echo strtoupper(substr($user_info, 0, 1)); ?>
                                                </div>
                                                <?php echo esc_html( $user_info ); ?>
                                            </div>
                                        </td>
                                        <td><code style="background:#F3F4F6; padding:2px 6px; border-radius:4px; font-size:12px;"><?php echo esc_html( $log->ip_address ); ?></code></td>
                                        <td style="color:#4B5563;"><?php echo esc_html( $log->message ); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php else : ?>
                                    <tr><td colspan="5" style="text-align:center; padding:40px; color:#9CA3AF;">No security logs found.</td></tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination Controls -->
                    <div class="hrm-date-pagination">
                         <div class="hrm-pagination-controls left">
                              <?php
                              $prev_url = '?page=hrm-settings&tab=security&week=' . ($week_offset + 1);
                              if ( $employee_filter ) {
                                  $prev_url .= '&employee_filter=' . $employee_filter;
                              }
                              ?>
                              <a href="<?php echo esc_url($prev_url); ?>" class="hrm-page-btn">
                                  <span class="dashicons dashicons-arrow-left-alt2"></span> Previous Week
                              </a>
                         </div>
                         
                         <div class="hrm-current-week-label">
                              <span class="dashicons dashicons-calendar-alt"></span>
                              <?php 
                                echo date('M d', $target_week_start) . ' - ' . date('M d, Y', $target_week_end); 
                              ?>
                         </div>
                         
                         <div class="hrm-pagination-controls right">
                              <?php
                              $next_url = '?page=hrm-settings&tab=security&week=' . max(0, $week_offset - 1);
                              if ( $employee_filter ) {
                                  $next_url .= '&employee_filter=' . $employee_filter;
                              }
                              ?>
                              <a href="<?php echo esc_url($next_url); ?>" 
                                 class="hrm-page-btn <?php echo ($week_offset === 0) ? 'disabled' : ''; ?>">
                                  Next Week <span class="dashicons dashicons-arrow-right-alt2"></span>
                              </a>
                         </div>
                    </div>
                </div>
            <?php endif; ?>
            </div>
        </div>
        
        <?php if ( $active_tab === 'general' ) : ?>
        <script>
        jQuery(document).ready(function($) {
            // Logo Upload Handler
            var logoFrame;
            
            $('#hrm-upload-logo-btn').on('click', function(e) {
                e.preventDefault();
                
                // Check if wp.media is available
                if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
                    alert('Media uploader not available. Please refresh the page and try again.');
                    return;
                }
                
                // If the media frame already exists, reopen it.
                if (logoFrame) {
                    logoFrame.open();
                    return;
                }
                
                // Create the media frame.
                logoFrame = wp.media({
                    title: 'Select or Upload Company Logo',
                    button: {
                        text: 'Use this logo'
                    },
                    multiple: false,
                    library: {
                        type: 'image'
                    }
                });
                
                // When an image is selected, run a callback.
                logoFrame.on('select', function() {
                    var attachment = logoFrame.state().get('selection').first().toJSON();
                    $('#hrm_company_logo_id').val(attachment.id);
                    $('#hrm-logo-preview').html('<img src="' + attachment.url + '" alt="Company Logo" style="max-width: 200px; max-height: 80px; width: auto; height: auto;">');
                    $('#hrm-upload-logo-btn').html('<span class="dashicons dashicons-upload" style="vertical-align: middle; margin-right: 6px;"></span> Change Logo');
                    
                    // Add remove button if not exists
                    if ($('#hrm-remove-logo-btn').length === 0) {
                        $('#hrm-upload-logo-btn').after('<button type="button" id="hrm-remove-logo-btn" class="hrm-btn-outline" style="padding: 10px 20px; color: #EF4444; border-color: #FCA5A5; margin-left: 12px;"><span class="dashicons dashicons-trash" style="vertical-align: middle; margin-right: 6px;"></span>Remove Logo</button>');
                    }
                });
                
                // Open the modal on click.
                logoFrame.open();
            });
            
            // Remove Logo Handler
            $(document).on('click', '#hrm-remove-logo-btn', function(e) {
                e.preventDefault();
                $('#hrm_company_logo_id').val('0');
                $('#hrm-logo-preview').html('<span style="color: #9CA3AF; font-size: 14px;"><span class="dashicons dashicons-format-image" style="font-size: 32px; width: 32px; height: 32px; display: block; margin: 0 auto 8px;"></span>No logo uploaded</span>');
                $('#hrm-upload-logo-btn').html('<span class="dashicons dashicons-upload" style="vertical-align: middle; margin-right: 6px;"></span> Upload Logo');
                $(this).remove();
            });
        });
        </script>
        <?php endif; ?>
        <?php
    }

    private static function get_dashboard_stats() {
        global $wpdb;
        $employee_count_obj = wp_count_posts( HRM_Setup::EMPLOYEE_POST_TYPE );
        $total_employees = 0;
        if (isset($employee_count_obj->publish)) {
            $total_employees += (int)$employee_count_obj->publish;
        }
        if (isset($employee_count_obj->draft)) {
            $total_employees += (int)$employee_count_obj->draft;
        }
        
        $exclude_sql = '';
        $exclude_progress_sql = '';
    
        // Test Mode exclusion removed - no longer used
    
        $pending_tasks = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_tasks WHERE status = 'pending' $exclude_sql" );
        $completed_tasks = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_tasks WHERE status = 'completed' $exclude_sql" );
    
        $recent_employees = get_posts([
            'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
            'numberposts' => 4,
            'orderby' => 'date',
            'order' => 'DESC'
        ]);
    
        $recent_tasks = $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}hrm_tasks ORDER BY created_at DESC LIMIT 4" );
    
        // Recent Progress Reports
        $recent_progress = $wpdb->get_results( "
            SELECT p.*, e.post_title as employee_name 
            FROM {$wpdb->prefix}hrm_daily_progress p
            LEFT JOIN {$wpdb->posts} e ON p.employee_id = e.ID
            ORDER BY p.date DESC, p.id DESC 
            LIMIT 5
        " );
    
        $today_date = current_time('Y-m-d');
        $reports_today = $wpdb->get_var( $wpdb->prepare( 
            "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_daily_progress WHERE date = %s $exclude_progress_sql", 
            $today_date 
        ));
    
        return [
            'total_employees' => $total_employees,
            'pending_tasks'   => $pending_tasks,
            'completed_tasks' => $completed_tasks,
            'recent_employees' => $recent_employees,
            'recent_tasks'    => $recent_tasks,
            'recent_progress' => $recent_progress,
            'reports_today'   => $reports_today
        ];
    }

    public static function render_dashboard() {
        $stats = self::get_dashboard_stats();
        $current_user = wp_get_current_user();
        ?>
        <div class="wrap hrm-wrap hrm-modern-dashboard">
            <!-- Header Section -->
            <div class="hrm-header-section">
                <div class="hrm-header-left">
                    <h1>Dashboard</h1>
                    <p>Plan, prioritize, and manage your HR tasks with ease.</p>
                </div>
                <div class="hrm-header-right" style="display:flex; align-items:center; gap:10px;">
                    <a href="<?php echo admin_url('admin.php?page=hrm-employees&action=add_new'); ?>" class="hrm-btn-primary">+ Add Employee</a>
                    <a href="<?php echo admin_url('admin.php?page=hrm-tasks'); ?>" class="hrm-btn-outline">Manage Tasks</a>
                </div>
            </div>
    
            <!-- Stats Row -->
            <div class="hrm-stats-row">
                <a href="<?php echo admin_url('admin.php?page=hrm-employees'); ?>" class="hrm-stat-card dark-green">
                    <div class="stat-header">
                        <span>Total Employees</span>
                        <span class="dashicons dashicons-groups" style="opacity: 0.8;"></span>
                    </div>
                    <div class="stat-number"><?php echo esc_html( $stats['total_employees'] ); ?></div>
                    <div class="stat-footer">
                        <span class="badge-green">Active Workforce</span>
                    </div>
                </a>

                <a href="<?php echo admin_url('admin.php?page=hrm-tasks&status=pending'); ?>" class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Pending Tasks</span>
                        <span class="dashicons dashicons-hourglass" style="color: var(--warning-color);"></span>
                    </div>
                    <div class="stat-number"><?php echo esc_html( $stats['pending_tasks'] ); ?></div>
                    <div class="stat-footer">
                        <span class="text-muted">Needs attention</span>
                    </div>
                </a>
                <a href="<?php echo admin_url('admin.php?page=hrm-tasks&status=completed'); ?>" class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Completed Tasks</span>
                        <span class="dashicons dashicons-yes-alt" style="color: var(--success-color);"></span>
                    </div>
                    <div class="stat-number"><?php echo esc_html( $stats['completed_tasks'] ); ?></div>
                    <div class="stat-footer">
                        <span class="text-muted">Finished this month</span>
                    </div>
                </a>
            </div>
    
            <!-- Main Grid -->
            <div class="hrm-main-grid">
                <!-- Left Column -->
                <div class="hrm-col-left">
                    <!-- Team Collaboration -->
                    <div class="hrm-card hrm-team-card">
                        <div class="card-header">
                            <h3>Team Members</h3>
                            <a href="<?php echo admin_url('admin.php?page=hrm-employees'); ?>" class="hrm-btn-outline btn-sm">View All</a>
                        </div>
                        <ul class="hrm-team-list">
                            <?php foreach($stats['recent_employees'] as $emp): 
                                $position = get_post_meta( $emp->ID, 'hrm_position', true ) ?: 'Employee';
                            ?>
                            <li class="team-item hrm-clickable-row" onclick="window.location='<?php echo admin_url('admin.php?page=hrm-employee-profile&id='.$emp->ID); ?>'">
                                <div class="member-avatar"><?php echo strtoupper(substr($emp->post_title, 0, 1)); ?></div>
                                <div class="member-info">
                                    <div class="member-name"><?php echo esc_html($emp->post_title); ?></div>
                                    <div class="member-role"><?php echo esc_html($position); ?></div>
                                </div>
                                <span class="dashicons dashicons-arrow-right-alt2" style="color: #ccc;"></span>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                </div>
    
                <!-- Right Column -->
                <div class="hrm-col-right">
                    <!-- Task Summary -->
                    <div class="hrm-card hrm-task-summary">
                        <div class="card-header">
                            <h3>Recent Tasks</h3>
                            <a href="<?php echo admin_url('admin.php?page=hrm-tasks'); ?>" class="hrm-btn-outline btn-sm">View All</a>
                        </div>
                         <?php if ( $stats['recent_tasks'] ) : ?>
                            <ul class="hrm-task-list-widget">
                                <?php foreach ( $stats['recent_tasks'] as $task ) : ?>
                                    <li class="task-item-widget">
                                        <div class="task-icon <?php echo $task->status; ?>"></div>
                                        <div class="task-info">
                                            <span class="task-title"><?php echo esc_html($task->title); ?></span>
                                            <span class="task-date">Due: <?php echo date('M d', strtotime($task->due_date)); ?></span>
                                        </div>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php else: ?>
                            <p class="hrm-no-data">No active tasks.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            // Modal close handlers
            $('.hrm-close-modal').on('click', function() {
                $('.hrm-modal').fadeOut();
            });

            // Close on outside click
            $(window).on('click', function(e) {
                if ($(e.target).is('.hrm-modal')) {
                    $('.hrm-modal').fadeOut();
                }
            });
            
            // Security Employee Filter Handler
            $('#apply-security-filter').on('click', function() {
                var employeeId = $('#security-employee-filter').val();
                var weekOffset = new URLSearchParams(window.location.search).get('week') || '0';
                var url = '?page=hrm-settings&tab=security&week=' + weekOffset;
                
                if (employeeId) {
                    url += '&employee_filter=' + employeeId;
                }
                
                window.location.href = url;
            });
            
            // Allow Enter key to apply security filter
            $('#security-employee-filter').on('keypress', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    $('#apply-security-filter').click();
                }
            });
        });
        </script>
        <?php
    }
}